#' Robust decomposition
#' 
#' Given a covariance matrix \code{v}, the function computes
#' a decomposition of \code{v} in a robust way.  If the
#' decomposition is \code{dvi}, then \code{dvi \%*\% v \%*\% t(dvi)}
#' should be the identity matrix.
#' 
#' @param v A square numerical matrix.
#' @param method The decomposition method.  Valid choices are
#' \code{"fast"} (try cholesky, then eigen, then singular value decomposition),
#' \code{"chol"} (cholesky), \code{"eigen"} (eigen), and 
#' \code{"svd"} (singular value decomposition).
#' @return A matrix
#' @export
#' @examples
#' # generate positive definite matrix
#' v <- crossprod(matrix(rnorm(25), nrow = 5))
#' 
#' # check cholesky
#' dvic <- RobustDecomp(v, method = "c")
#' all.equal(dvic %*% v %*% t(dvic), diag(5))
#' 
#' # check eigen
#' dvie <- RobustDecomp(v, method = "e")
#' all.equal(dvie %*% v %*% t(dvie), diag(5))
#' 
#' # check svd
#' dvis <- RobustDecomp(v, method = "s")
#' all.equal(dvis %*% v %*% t(dvis), diag(5))
RobustDecomp <- function(v, method = "f") {
  if (!is.matrix(v)) stop("v must be a matrix")
  if (nrow(v) != ncol(v)) stop("v must be a square matrix")
  method <- match.arg(method, c("fast", "chol", "eigen", "svd"))
  if (method == "fast") {
    dvi <- try(chol(solve(v)), silent = TRUE)
    if (class(dvi) == "try-error") {
      dvi <- try(t(SpatialTools::decomp.cov(solve(v), method = "eigen")), silent = TRUE)
    } 
    if (class(dvi) == "try-error") {
      dvi <- t(SpatialTools::decomp.cov(solve(v), method = "svd"))
    }
  } else if (method == "chol") {
    dvi <- chol(solve(v))
  } else if (method == "eigen") {
    dvi <- t(SpatialTools::decomp.cov(solve(v), method = "eigen"))
  } else {
    dvi <- t(SpatialTools::decomp.cov(solve(v), method = "svd"))
  }
  dvi
}
