library(mvtnorm) # for computing aic
library(ggplot2)
library(ggthemes) # for theme_base
library(maps) # for world map
library(abind) 

# national boundaries
wor = map("world", plot = FALSE)

setwd("~/Dropbox/UCD_Files/Research/In Progress/narccap/analysis")
# load relevant data
load("tmax_summer_covar.rda")
load("cope_diagnostics_bias2_glsmods.rda")
load("fit_details.rda")

# aic storage
aic_apl = aic_apl1 = aic_apl2 = aic_ipl1 = aic_ipl2 = aic_asl = aic_asl1 = aic_asl2 = aic_isl1 = aic_isl2 = vector("list", nrow(idx))

# compute aic
myaic = function(x){
  -2 * dmvnorm(x$y, x$fitted.values, x$v, log = TRUE) + 2 * length(x$coef)
}

# compute aics for each locations
aic_apl = sapply(lmod_apl, myaic)
aic_apl1 = sapply(lmod_apl1, myaic)
aic_apl2 = sapply(lmod_apl2, myaic)
aic_ipl1 = sapply(lmod_ipl1, myaic)
aic_ipl2 = sapply(lmod_ipl2, myaic)
aic_asl = sapply(lmod_asl, myaic)
aic_asl1 = sapply(lmod_asl1, myaic)
aic_asl2 = sapply(lmod_asl2, myaic)
aic_isl1 = sapply(lmod_isl1, myaic)
aic_isl2 = sapply(lmod_isl2, myaic)

# convert to array format for plotting
aic_apl_array = aic_apl1_array = aic_apl2_array = aic_ipl1_array = aic_ipl2_array = aic_asl_array = aic_asl1_array = aic_asl2_array = aic_isl1_array = aic_isl2_array = matrix(NA, nrow = length(lon), ncol = length(lat))
for (j in seq_along(aic_apl)) {
  aic_apl_array[idx[j,1],idx[j,2]] = aic_apl[j]
  aic_apl1_array[idx[j,1],idx[j,2]] = aic_apl1[j]
  aic_apl2_array[idx[j,1],idx[j,2]] = aic_apl2[j]
  aic_ipl1_array[idx[j,1],idx[j,2]] = aic_ipl1[j]
  aic_ipl2_array[idx[j,1],idx[j,2]] = aic_ipl2[j]
  aic_asl_array[idx[j,1],idx[j,2]] = aic_asl[j]
  aic_asl1_array[idx[j,1],idx[j,2]] = aic_asl1[j]
  aic_asl2_array[idx[j,1],idx[j,2]] = aic_asl2[j]
  aic_isl1_array[idx[j,1],idx[j,2]] = aic_isl1[j]
  aic_isl2_array[idx[j,1],idx[j,2]] = aic_isl2[j]
}

xy = expand.grid(lon, lat)
wordf = data.frame(x = wor$x, y = wor$y)

# make blank theme for ggplot2
theme_blank <- theme_base() + 
  theme(axis.line = element_blank(),
        axis.text.x = element_blank(),
        axis.text.y = element_blank(),
        axis.ticks = element_blank(),
        axis.title.x = element_blank(),
        axis.title.y = element_blank(),
        panel.background = element_blank(),
        panel.border = element_blank(),
        panel.grid.major = element_blank(),
        panel.grid.minor = element_blank(),
        plot.background = element_blank())

# determine best model in terms of AIC
c1 = ifelse(aic_apl_array < aic_asl_array, "APL", "ASL")
c2 = ifelse(aic_apl1_array < aic_ipl1_array, "APL", "IPL")
c3 = ifelse(aic_apl2_array < aic_ipl2_array, "APL", "IPL")
c4 = ifelse(aic_ipl1_array < aic_isl1_array, "IPL", "ISL")
c5 = ifelse(aic_ipl2_array < aic_isl2_array, "IPL", "ISL")

# stack aics statistics for mini-experiments, determine which is min
me1stack = abind(aic_apl1_array, aic_ipl1_array, aic_asl1_array, aic_isl1_array, along = 3)
me2stack = abind(aic_apl2_array, aic_ipl2_array, aic_asl2_array, aic_isl2_array, along = 3)

# make function to determine which model has the smallest AIC
wmin = function(x){
  mymin = which.min(x)
  if (length(mymin) == 0) mymin = 5
  c("APL", "IPL", "ASL", "ISL", NA)[mymin]
}
# smallest AIC model for each mini-experiment
me1_wmin = unname(apply(me1stack, 1:2, wmin))
me2_wmin = unname(apply(me2stack, 1:2, wmin))

# stack best model names, format for plotting
q3 = (c(c(c1), c(me1_wmin), c(me2_wmin)))
name3 = name = rep(c("(a) all data", "(b) mini-experiment 1", "(c) mini-experiment 2"), each = length(c1))
fq3 = factor(q3)
df3 = data.frame(lon = rep(xy[,1], 3), 
                 lat = rep(xy[,2], 3), 
                 z = factor(q3), 
                 name = name3)
df3b = na.omit(df3) # clean out NAs
pdf("fig05_aic_plots_update.pdf", height = 2.5, width = 6)
ggplot(df3b, aes(x = lon - 360, y = lat)) + 
  geom_tile(aes(fill = z), data = df3b) + 
  theme_base() + 
  facet_wrap(~ name) + 
  geom_path(aes(x = x, y = y), wordf, col = "black") + 
  xlim(235.25 - 360, 292.75 - 360) + 
  ylim(25.25, 49.75) + 
  coord_map(projection = "lambert", lat0 = 33, lat1 = 45) + 
  theme_blank +  
  theme(legend.position = "bottom", legend.direction = "horizontal") + 
  guides(fill = guide_legend(title = NULL)) + 
  scale_fill_manual(values = c("#a6cee3", "#1f78b4", "#b2df8a", "#33a02c"), na.value = "white")
dev.off()
