library(magic)
library(nlme)
library(MASS)
library(fdrtool)
library(ggplot2)

# complete v estimated for gls_fit
vest = function(x, df, p) {
  tdf = df
  tdf$r = x$residuals
  stdf = split(tdf, tdf$gr)
  # vs = mapply(gls_v, data = stdf, p = p)
  vs = vector("list", length(stdf))
  for (j in seq_along(vs)) vs[[j]] = gls_v(stdf[[j]], p[j])
  v = Reduce(adiag, vs)
  obs_num = c(unlist(sapply(stdf, row.names)))
  if (!all.equal(row.names(tdf), obs_num, check.attributes = FALSE)) {
    stop("row.names(tdf) != obs_num")
  }
  return(v)
}

# invert a covariance matrix v, factor to a matrix
# dvi such that dvi %*% v %*% t(dvi) = I.  Do this with
# error handling
robust_dvi <- function(v) {
  dvi <- try(chol(solve(v)), silent = TRUE)
  if (class(dvi) == "try-error") {
    dvi = t(SpatialTools::decomp.cov(solve(v)))
  }
  dvi
}

# fit general linear model, estimating covariance in the process
gls_fit = function(f, df, p){
  x = lm(f, df)
  v = vest(x, df, p)
  glsmod = lm.gls(f, df, W = v, inverse = TRUE)
  dvi <- robust_dvi(v)
  glsmod$sresiduals = dvi %*% glsmod$residuals
  glsmod$v = v
  glsmod$y = nlme::getResponse(df, f)
  glsmod$gr = df$gr
  glsmod$gcm = df$gcm
  glsmod$rcm = df$rcm
  glsmod
} 

# estimate v for gls_fit
gls_v = function(data, p) {
  v = NULL
  while (is.null(v)) {
    try_fit = "try-pass"
    if (p > 0) {
      try_fit = try(tgls <- gls(r ~ 1, data = data, correlation = corARMA(form = ~ 1, p = p, q = 0)))
      if (class(try_fit) != "try-error") {
        v = corMatrix(tgls$modelStruct$corStruct) * tgls$sigma^2
      }
    } else if (p == 1) {
      try_fit = try(tgls <- gls(r ~ 1, data = data, correlation = corAR1(form = ~ 1)))
      if (class(try_fit) != "try-error") {
        v = corMatrix(tgls$modelStruct$corStruct) * tgls$sigma^2
      }
    } else {
      tgls = lm(r ~ 1, data = data)
      sigma = summary(tgls)$sigma
      v = sigma^2*diag(nrow(data))
    }
    if (class(try_fit) == "try-error") p = p - 1
  }
  return(v)
}

# estimate proportion of null hypotheses from set of p-values
eta0 = function(x) {
  tapply(x, names(x), fdrtool::pval.estimate.eta0, diagnostic.plot = FALSE)
}

# estimate empirical cdf
myecdf = function(x) sapply(seq(0.01, 0.99, len = 99), function(y) mean(x <= y, na.rm = TRUE))
lbp = function(x){
  data.table::rbindlist(lapply(x, function(x) data.frame(p = x, model = names(x))))
}

# estimate epirical cdf to a list of values
cdf = function(x) {
  tapply(x, names(x), myecdf)
}

# plot cdf by gcm/rcm combination
plot_cdf = function(x) {
  xdf = as.data.frame(t(do.call(rbind, x)))
  xdf = tidyr::gather(xdf)
  xdf$p = seq(0.01, 0.99, len = 99)
  ggplot(xdf, aes(x = p, y = value)) + geom_line() + 
    facet_wrap(~ key) + geom_abline(intercept = 0, slope = 1)
}

# Ljung-Box test for independence of time series
# Done for each gcm/rcm combination
lb_test = function(x){
  lbtests = tapply(x$sresid, factor(x$gr), stats::Box.test, type = "Ljung-Box", lag = 10)
  sapply(lbtests, getElement, "p.value")
}

# Apply Ljung_Box tests to a list of fits from gls_fit
# Return in convenient format
lb_lmod = function(obj) { 
  unlist(lapply(obj, lb_test))
}

# plot tempdata
plot_tempdata = function(tempdata) {
  ggplot(tempdata, aes(x = yr, y = temp)) + geom_point() + facet_wrap(~ gr, scale = "free")
}

# plot results of gls_fit
plot_gls_fit = function(glsfit, type = "r") {
  nyr = table(glsfit$gr)
  myyr = c(unlist(sapply(nyr, seq_len)))
  mydf = data.frame(yr = yr, resid = glsfit$residuals, 
                    sresid = glsfit$sresiduals,
                    gr = glsfit$gr)
  if (type == "r") {
    ggplot(mydf, aes(x = yr, y = resid)) + geom_point() + facet_wrap(~ gr, scale = "free")
  } else {
    ggplot(mydf, aes(x = yr, y = sresid)) + geom_point() + facet_wrap(~ gr, scale = "free")
  }
}

# # take list of gls_fits, and create 3d array
# # using binded estimates of v
# v_bind <- function(x) {
#   v_array = array(0, dim = c(dim(x[[1]]$v), length(x)))
#   for (i in seq_len(length(x))) v_array[,,i] = x[[i]]$v
#   v_array
# }

# do standard hypothesis test for linear combination specified by w, return estimates at significant locations
gls_lh = function(Z, X, w, V, mask = NULL, level = NULL, alpha = 0.1){
  x = Z$x
  y = Z$y
  Y = Z$z
  n = dim(Y)[3]
  nloc <- length(x) * length(y)

  p <- ncol(X)

  pvals <- matrix(0, length(x), length(y)) # Estimate of the function of
  for (i in 1:length(x)) {
    for (j in 1:length(y)) {
      ytemp <- Y[i, j,]
      if (sum(is.na(ytemp)) == length(ytemp)) { # deal with problem if ytemp is all NAs
        pvals[i, j] = NA
      }else{
        # T = chol(solve(V[i,j,,]))
        T = dvi <- robust_dvi(V[i,j,,])

        Ty = T %*% ytemp
        TX = T %*% X

        m2 = lm(Ty ~ TX - 1)
        vbeta2 = vcov(m2)/summary(m2)$sigma^2
        q = car::linearHypothesis(m2, w)
        pvals[i, j] = ifelse(q$`Pr(>F)`[2] < alpha, 1, NA) * ((w %*% coef(m2)))[1,1]
      }
    }
  }
  pvals
}
# 
# # do standard hypothesis test, return se estimates at significant locations
# gls_lh_se = function(Z, X, w, V, mask = NULL, level = NULL, alpha = 0.1){
#   x = Z$x
#   y = Z$y
#   Y = Z$z
#   n = dim(Y)[3]
#   nloc <- length(x) * length(y)
#   
#   p <- ncol(X)
#   
#   pvals <- matrix(0, length(x), length(y)) # Estimate of the function of 
#   for (i in 1:length(x)) {
#     for (j in 1:length(y)) {
#       ytemp <- Y[i, j,]
#       if (sum(is.na(ytemp)) == length(ytemp)) { # deal with problem if ytemp is all NAs
#         pvals[i, j] = NA
#       }else{
#         T = chol(solve(V[i,j,,]))
#         
#         Ty = T %*% ytemp
#         TX = T %*% X
#         
#         m2 = lm(Ty ~ TX - 1)
#         vbeta2 = vcov(m2)/summary(m2)$sigma^2
#         q = car::linearHypothesis(m2, w)
#         pvals[i, j] = ifelse(q$`Pr(>F)`[2] < alpha, 1, NA) * ((t(w) %*% vbeta2 %*% w))[1,1]
#       }
#     }
#   }
#   pvals
# }
# 
# get significant rejection region for cope models
get.rej2 = function(x)
{
  a = x$a_MB
  Acz = x$mask*x$mu_hat
  zna = x$norm_est <= a & x$norm_est >= -a
  which_na = which(zna == TRUE, arr.ind = TRUE)
  zmask = matrix(1, nrow = nrow(x$mu_hat), ncol = ncol(x$mu_hat))
  zmask[which_na] = NA
  z = Acz * zmask
  return(z)
}


